/*
** Made by fabien le mentec <texane@gmail.com>
** 
** Started on  Tue Nov 17 04:21:01 2009 fabien le mentec
** Last update Tue Dec 13 10:24:08 2011 fabien le mentec
*/



#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>


#include "zusb.h"
#include "debug.h"
#include "slosyn.h"
#include "slosyn_types.h"




/* utility macro */

#define GOTO_ERROR(E)	\
  do {			\
    error = E;		\
    goto on_error;	\
  } while (0)


/* allocation wrappers */

static slosyn_handle_t* alloc_slosyn_handle(void)
{
  slosyn_handle_t* handle = malloc(sizeof(slosyn_handle_t));
  if (handle == NULL)
    return NULL;

  handle->usb_handle = NULL;
  
  return handle;
}


static void free_slosyn_handle(slosyn_handle_t* handle)
{
  free(handle);
}


/* translate libusb error */

static slosyn_error_t __attribute__((unused)) usb_to_slosyn_error(int ue)
{
  return SLOSYN_ERROR_LIBUSB;
}

u_int16_t  vendrID ;
u_int16_t  lectrID ;

/* is the usb device a slosyn tape reader */

static unsigned int is_slosyn_device(const struct usb_device *dev)
{
u_int16_t  vendrID0 ;
u_int16_t  lectrID0 ;

	vendrID0 = dev->descriptor.idVendor ;
	lectrID0 = dev->descriptor.idProduct ;
	
//	if (dev->descriptor.idVendor != SLOSYN_USB_VENDOR_ID)
	if (dev->descriptor.idVendor != vendrID)
	return 0;

//	if (dev->descriptor.idProduct != SLOSYN_USB_PRODUCT_ID)
	if (dev->descriptor.idProduct != lectrID)
    return 0;

  return 1;
}


static slosyn_error_t send_recv_cmd(slosyn_handle_t* handle, slosyn_cmd_t* cmd)
{
  /* timeouts in ms */
#define CONFIG_DEFAULT_TIMEOUT (2000)

  int error = SLOSYN_ERROR_SUCCESS;
  int usberr;

  usberr = usb_bulk_write
    (handle->usb_handle, handle->ep_req, (void*)&cmd->req,
     (int)sizeof(cmd->req), CONFIG_DEFAULT_TIMEOUT);
  if (usberr < 0)
  {
    GOTO_ERROR(SLOSYN_ERROR_LIBUSB);
  }

  usberr = usb_bulk_read
    (handle->usb_handle, handle->ep_rep, (void*)&cmd->rep,
     (int)sizeof(cmd->rep), CONFIG_DEFAULT_TIMEOUT);
  if (usberr < 0)
  {
    GOTO_ERROR(SLOSYN_ERROR_LIBUSB);
  }

 on_error:
  return error;
}


/* forward decls */

static slosyn_error_t open_slosyn_usb_handle(usb_dev_handle**, int);
static void close_slosyn_usb_handle(usb_dev_handle*);

//int find_slosyn_device(void);

slosyn_error_t send_recv_cmd_or_reopen(slosyn_handle_t* handle, slosyn_cmd_t* cmd)
{
  /* send_recv_cmd or reopen handle on libusb failure */

  slosyn_error_t error = send_recv_cmd(handle, cmd);
  if (error != SLOSYN_ERROR_SUCCESS)
  {
    if (find_slosyn_device() == -1)
      GOTO_ERROR(SLOSYN_ERROR_NOT_FOUND);

    /* reopen the usb handle */

    close_slosyn_usb_handle(handle->usb_handle);

    error = open_slosyn_usb_handle(&handle->usb_handle, 0);
    if (error != SLOSYN_ERROR_SUCCESS)
      GOTO_ERROR(error);

    /* resend the command */

    error = send_recv_cmd(handle, cmd);
  }

 on_error:
  return error;
}


/* endianness */

static int is_mach_le = 0;

static inline int get_is_mach_le(void)
{
  const uint16_t magic = 0x0102;

  if ((*(const uint8_t*)&magic) == 0x02)
    return 1;

  return 0;
}

static inline uint16_t le16_to_mach(uint16_t n)
{
  if (!is_mach_le)
    n = ((n >> 8) & 0xff) | ((n & 0xff) << 8);

  return n;
}


static void close_slosyn_usb_handle(usb_dev_handle* usb_handle)
{
  usb_close(usb_handle);
}

static slosyn_error_t open_slosyn_usb_handle(usb_dev_handle** usb_handle, int enum_only)
{
  /* enum_only is used when we only wants to find
     the slosyn device. */

  slosyn_error_t error;
  struct usb_bus* bus;

  *usb_handle = NULL;

  usb_find_busses();
  usb_find_devices();

  for (bus = usb_busses; bus != NULL; bus = bus->next)
  {
    struct usb_device* dev;
    for (dev = bus->devices; dev != NULL; dev = dev->next)
    {
      if (is_slosyn_device(dev))
      {
	if (enum_only)
	  GOTO_ERROR(SLOSYN_ERROR_SUCCESS);

	*usb_handle = usb_open(dev);
	if (*usb_handle == NULL)
	  GOTO_ERROR(SLOSYN_ERROR_LIBUSB);

	if (usb_set_configuration(*usb_handle, 2))
	{
	  GOTO_ERROR(SLOSYN_ERROR_LIBUSB);
	}

	if (usb_claim_interface(*usb_handle, 0))
	{
	  GOTO_ERROR(SLOSYN_ERROR_LIBUSB);
	}

	return SLOSYN_ERROR_SUCCESS;
      }
    }
  }

  /* not found */

  error = SLOSYN_ERROR_NOT_FOUND;

 on_error:

  if (*usb_handle != NULL)
  {
    usb_close(*usb_handle);
    *usb_handle = NULL;
  }

  return error;
}

/*
static inline size_t min(size_t a, size_t b)
{
  return a < b ? a : b; 
}
 // */

/* exported functions ***********************************************************************************/

/*void slosyn_cleanup(void)
{
} */

/* find slosyn device wrapper */

int find_slosyn_device(void)
{
  /* return 0 if slosyn device found */

  usb_dev_handle* dummy_handle = NULL;
  int res = 0;

  const slosyn_error_t error = open_slosyn_usb_handle(&dummy_handle, 1);
  if (error == SLOSYN_ERROR_NOT_FOUND)
    res = -1;

  return res;
}


slosyn_error_t slosyn_initialize(void)
{
  is_mach_le = get_is_mach_le();

  usb_init();

  return SLOSYN_ERROR_SUCCESS;
}


slosyn_error_t slosyn_open(slosyn_handle_t** slosyn_handle)
{
  slosyn_error_t error;
  usb_dev_handle* usb_handle = NULL;
  
  *slosyn_handle = NULL;

  /* open usb device */

  error = open_slosyn_usb_handle(&usb_handle, 0);
  if (error != SLOSYN_ERROR_SUCCESS)
    goto on_error;

  /* open and init the device */

  *slosyn_handle = alloc_slosyn_handle();
  if (*slosyn_handle == NULL)
    GOTO_ERROR(SLOSYN_ERROR_MEMORY);

  (*slosyn_handle)->usb_handle = usb_handle;

  (*slosyn_handle)->ep_req = SLOSYN_USB_EP_REQ | USB_ENDPOINT_OUT;
  (*slosyn_handle)->ep_rep = SLOSYN_USB_EP_REP | USB_ENDPOINT_IN;

  error = SLOSYN_ERROR_SUCCESS;
  
 on_error:

  return error;
}


void slosyn_close(slosyn_handle_t* handle)
{
  if (handle->usb_handle != NULL)
    close_slosyn_usb_handle(handle->usb_handle);

  free_slosyn_handle(handle);
}

